#ifdef WIN32

#include "MeetingMgrControllerImplWin.h"
#include "csf/logger/CSFLogger.hpp"
#include "jcfcoreutils/StringUtils.h"
#include <csfunified/framework/UnifiedFactory.h>
#include "csfunified/framework/ServicesDispatcher.h"
#include "csf/Task.hpp"
#include "jcfcoreutils/FunctionTask.h"
#include "csfunified/framework/FunctorTask.h"
#include "csfunified/services/impl/BaseToImplCast.h"
#include "services/impl/MeetingServiceImpl.h"
#include "services/impl/cmr/CMRControllerMgr.h"

#include "featuresets/adapters/Dependence.h"
#include "boost/bind.hpp"

#include "services/impl/MeetingInfoImpl.h"

#include "services/impl/MeetingServiceObjectManager.h"

#include "sdkwrapper/win32/autoupgrade/jabber_sdk_wrap_impl.h"
#include "autoupgrade/MeetingSDKMgr.h"

using namespace JM_SDK_WRAP;

namespace CSFUnified
{
	static CSFLogger* MeetingMgrControllerImplLogger = CSFLogger_getLogger("MeetingService-MeetingMgrControllerImpl");

	MeetingMgrControllerImpl::MeetingMgrControllerImpl(UnifiedFactoryPtr unifiedFactory) : 
		MeetingMgrControllerBaseImpl(unifiedFactory), m_unifiedFactory(unifiedFactory)
	{
		m_pMeetingMgr = NULL;
		m_pIMMgr = NULL;
		started = false;
	}

	void MeetingMgrControllerImpl::start()
	{
		CDependence::getInstance()->start();
		
		m_pMeetingMgr = CDependence::getInstance()->getMeetingMgrHandle();
		if (NULL != m_pMeetingMgr)
		{
			SMART_PTR_NS::shared_ptr<MeetingServiceImpl> meetingService = baseToImplCast<MeetingService, MeetingServiceImpl>(this->unifiedFactory->getService<MeetingService>());
			int eType = (int)meetingService->getIntegrationCalendarType();
			JabberMeeting::ConOIServieType conOIType = (JabberMeeting::ConOIServieType)eType;
			m_pMeetingMgr->Initialize(conOIType);

			JabberSDKWrapImpl::getInstance()->setJMMeetingMgrHandler(this);
		}

		m_pIMMgr = CDependence::getInstance()->getInstantMeetingMgrHandle();
		if (NULL != m_pIMMgr)
		{
			JabberSDKWrapImpl::getInstance()->setIInstantMeetingEvent(this);
		}

		if (MeetingSDKMgr::getInstance()->isAutoUpgradeFeatureEnabled())
		{
			MeetingSDKMgr::getInstance()->startUpgradeCheck();
		}
	}

	void MeetingMgrControllerImpl::stop()
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::stop");

		if (MeetingSDKMgr::getInstance()->isAutoUpgradeFeatureEnabled())
		{
			MeetingSDKMgr::getInstance()->stopUpgradeCheck();
		}

		if (NULL != m_pMeetingMgr)
		{
			m_pMeetingMgr->ReleaseResources();
		}

		if (NULL != m_pIMMgr)
		{
			JabberSDKWrapImpl::getInstance()->setIInstantMeetingEvent(NULL);
			m_pIMMgr = NULL;
		}

		CDependence::getInstance()->stop();
	}

#pragma region Implement Controller

	void MeetingMgrControllerImpl::registerMeetingCallback(SMART_PTR_NS::shared_ptr<CSFUnified::MeetingCallback> meetingCallback)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "DesktopSharingControllerImpl::registerDSCallback()");
		m_meetingCallback = meetingCallback.get();
	}

	void MeetingMgrControllerImpl::unregisterMeetingCallback(SMART_PTR_NS::shared_ptr<CSFUnified::MeetingCallback> meetingCallback)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "DesktopSharingControllerImpl::unregisterDSCallback()");
		m_meetingCallback = NULL;
	}

    bool MeetingMgrControllerImpl::isMeetingInProgress(std::string meetingID)
    {
        std::wstring wMeetingID = JCFCoreUtils::toWideString(meetingID);
        if (NULL != m_pIMMgr)
        {
            return m_pIMMgr->IsMeetingInProgress(wMeetingID) ? true : false;
        }
        return false;
    }
    
	unsigned long MeetingMgrControllerImpl::listMeetings(int year, int month, int day)
	{
		SYSTEMTIME stSelectedDay;
		stSelectedDay.wYear = year;
		stSelectedDay.wMonth = month;
		stSelectedDay.wDay = day;

		if (NULL != m_pMeetingMgr)
		{
			CDependence::getInstance()->GetJabberWerxCommonHelper()->SetListMeetingDate(&stSelectedDay);
			return m_pMeetingMgr->ListEvents(stSelectedDay);
		}

		return 0;
	}

	bool MeetingMgrControllerImpl::getOnlineSwitch()
	{
		BOOL bVal = FALSE;
		if (NULL != m_pMeetingMgr)
		{
			m_pMeetingMgr->GetOnlineSwitch(&bVal);
		}

		return bVal==TRUE;
	}

	bool MeetingMgrControllerImpl::getOfflineSwitch()
	{
		BOOL bVal = FALSE;
		if (NULL != m_pMeetingMgr)
		{
			m_pMeetingMgr->GetOfflineSwitch(&bVal);
		}

		return bVal == TRUE;
	}

	unsigned long MeetingMgrControllerImpl::refreshMeetings()
	{
		if (NULL != m_pMeetingMgr)
		{
			return m_pMeetingMgr->RefreshCalendar();
		}

		return 0;
	}

	bool MeetingMgrControllerImpl::todayChanged(bool bForceChanged)
	{
		if (NULL != m_pMeetingMgr)
		{
			CDependence::getInstance()->GetJabberWerxCommonHelper()->SetListMeetingDate();
			return SUCCEEDED(m_pMeetingMgr->TodayChanged(bForceChanged));
		}

		return 0;
	}

	unsigned long MeetingMgrControllerImpl::getMeetingDetail(std::string strEventID)
	{
		std::wstring wstrID = JCFCoreUtils::toWideString(strEventID);

		if (NULL != m_pMeetingMgr)
		{
			return m_pMeetingMgr->GetWbxMeetingDetail(wstrID);
		}

		return 0;
	}

	unsigned long MeetingMgrControllerImpl::viewMeetingById(std::string strMeetingID)
	{
		std::wstring wstrID = JCFCoreUtils::toWideString(strMeetingID);

		if (NULL != m_pMeetingMgr)
		{
			return m_pMeetingMgr->ViewEvent(wstrID);
		}

		return 0;
	}

	unsigned long MeetingMgrControllerImpl::launchMeetingById(std::string strEventID)
	{
		std::wstring wstrID = JCFCoreUtils::toWideString(strEventID);

		if (NULL != m_pMeetingMgr)
		{
			return m_pMeetingMgr->LaunchWebexMeeting(wstrID);
		}

		return 0;
	}

	MailServiceTypeEnum::MailServiceType MeetingMgrControllerImpl::getCurrentMailType()
	{
		int conType = MailServiceTypeEnum::NONE;

		if (NULL != m_pMeetingMgr)
		{
			JabberMeeting::ConOIServieType conOIType;
			m_pMeetingMgr->GetCurrentMailType(conOIType);
			conType = (int)conOIType;
		}

		return (MailServiceTypeEnum::MailServiceType)conType;
	}

	unsigned long MeetingMgrControllerImpl::sendEmail(std::string strTos, std::string strSubject, std::string strBody, std::string strMeetingKey)
	{
		std::wstring wstrSubject = JCFCoreUtils::toWideString(strSubject);
		std::wstring wstrTos = JCFCoreUtils::toWideString(strTos);
		std::wstring wstrBody = JCFCoreUtils::toWideString(strBody);
		std::wstring wstrMeetingKey = JCFCoreUtils::toWideString(strMeetingKey);

		if (NULL != m_pMeetingMgr)
		{
			return m_pMeetingMgr->SendMail(wstrSubject, wstrTos, wstrBody, wstrMeetingKey);
		}

		return 0;
	}

	void MeetingMgrControllerImpl::startCMRMeeting(bool isAutoStart)
	{
		if (NULL != m_pIMMgr)
		{
			m_pIMMgr->StartCMRMeeting(isAutoStart);
		}
	}

	void MeetingMgrControllerImpl::joinCMRMeeting(std::string siteUrl, std::string meetingKey, std::string mkgtk)
	{
		std::wstring wsSiteUrl = JCFCoreUtils::toWideString(siteUrl);
		std::wstring wsMeetingKey = JCFCoreUtils::toWideString(meetingKey);
		std::wstring wsMkgtk = JCFCoreUtils::toWideString(mkgtk);

		if (NULL != m_pIMMgr)
		{
			m_pIMMgr->JoinCMRMeeting(wsSiteUrl, wsMeetingKey, wsMkgtk);
		}
	}

	void MeetingMgrControllerImpl::getCMRInfo(std::string strMeetingKeyList)
	{
		std::wstring meetingKeyList = JCFCoreUtils::toWideString(strMeetingKeyList);
		if (NULL != m_pMeetingMgr)
		{
			m_pMeetingMgr->GetCMRInfo(meetingKeyList);
		}
	}

	void MeetingMgrControllerImpl::networkChanged(bool bConnected)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::networkChanged connected: " << bConnected);

		if (NULL != m_pMeetingMgr)
		{
			m_pMeetingMgr->NetworkChanged(bConnected);
		}

	}
#pragma endregion

#pragma region Instant Meeting

	bool MeetingMgrControllerImpl::isInMeeting()
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::IsInMeeting");
		if (NULL != m_pIMMgr)
		{
			return m_pIMMgr->IsInMeeting() ? true : false;
		}
		return false;
	}

	bool MeetingMgrControllerImpl::canStartMeeting()
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::CanStartMeeting");
		if (NULL != m_pIMMgr)
		{
			return m_pIMMgr->CanStartMeeting() ? true : false;
		}
		return false;
	}

	bool MeetingMgrControllerImpl::canJoinMeeting()
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::CanJoinMeeting");
		if (NULL != m_pIMMgr)
		{
			return m_pIMMgr->CanJoinMeeting(_T("")) ? true : false;
		}
		return false;
	}

	bool MeetingMgrControllerImpl::startInstantMeeting(std::string meetingName, SMART_PTR_NS::shared_ptr<std::vector<std::string> > vecInviteList)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::StartInstantMeeting");
		if (NULL != m_pIMMgr)
		{
			std::vector<InviteeInfo> vecUserList;
			for (std::vector<std::string>::iterator it = vecInviteList->begin(); it != vecInviteList->end(); it++)
			{
				InviteeInfo item;
				item.strValue = JCFCoreUtils::toWideString(*it);
				item.eType = InviteeInfo::TYPE_JID;
				vecUserList.push_back(item);
			}

			return m_pIMMgr->StartInstantMeeting(JCFCoreUtils::toWideString(meetingName), vecUserList, wstring(L"")) ? true : false;
		}
		return false;
	}

	bool MeetingMgrControllerImpl::inviteToInstantMeeting(std::string strBuddyName)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::InviteUser2InstantMeeting");
		if (NULL != m_pIMMgr)
		{
			return m_pIMMgr->Reinvite(JCFCoreUtils::toWideString(strBuddyName)) ? true : false;
		}
		return false;
	}

	bool MeetingMgrControllerImpl::acceptInstantMeeting(std::string strSessioName)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::Accept");
		if (NULL != m_pIMMgr)
		{
			return m_pIMMgr->Accept(JCFCoreUtils::toWideString(strSessioName)) ? true : false;
		}
		return false;
	}

	bool MeetingMgrControllerImpl::declineInstantMeeting(std::string strSessioName)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::Decline");
		if (NULL != m_pIMMgr)
		{
			return m_pIMMgr->Decline(JCFCoreUtils::toWideString(strSessioName)) ? true : false;
		}
		return false;
	}

	int MeetingMgrControllerImpl::getInstantMeetingLastErrCode()
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::GetLastErrCode");
		if (NULL != m_pIMMgr)
		{
			std::wstring strErrMsg;
			return m_pIMMgr->GetLastError(strErrMsg);
		}
		return 0;
	}

	void MeetingMgrControllerImpl::editInstantMeetingSetting()
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OpenOneClickSetting");
		if (NULL != m_pIMMgr)
		{
			m_pIMMgr->OpenOneClickSetting();
		}
	}

	bool MeetingMgrControllerImpl::openMeetingDetailPage(std::string pszJoinUrl)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OpenMeetingDetailPage");
		std::wstring strURL = JCFCoreUtils::toWideString(pszJoinUrl);
		if (NULL != m_pIMMgr)
		{
			return m_pIMMgr->OpenMeetingDetailPage(strURL.c_str()) ? true : false;
		}
		return false;
	}

	bool MeetingMgrControllerImpl::startJoinMeetingByUrl(std::string pszJoinURl)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::StartJoinMeetingByUrl");
		std::wstring strURL = JCFCoreUtils::toWideString(pszJoinURl);
		if (NULL != m_pIMMgr)
		{
			return m_pIMMgr->StartJoinMeetingByUrl(strURL.c_str()) ? true : false;
		}
		return false;
	}
#pragma endregion

	void MeetingMgrControllerImpl::OnListOneDayEvents(std::wstring const& strEvents)
	{
		//Update meeting to list webex meeting on status and locked in onUpdateMeetingList
		CDependence::getInstance()->GetJabberWerxCommonHelper()->onUpdateMeetingList(strEvents);

		std::string strContent = JCFCoreUtils::toString(strEvents);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnListOneDayEvents");

		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				// Mingfeng 2015.8.28: strEvents is XML content for all meetings to list. To unify meeting 
				// service interface and be compatible with UI layer code, temporarily set the content to newly
				// added MeetingEventXML in the first node of meetingList vector.
				MeetingInfoImpl * pObj = new MeetingInfoImpl(unifiedFactory);
				pObj->setMeetingEventXML(strContent);
				SMART_PTR_NS::shared_ptr<MeetingInfoImpl> sharedObj;
				sharedObj.reset(pObj);

				std::vector<SMART_PTR_NS::shared_ptr<MeetingInfo> >* pMeetingVec = new std::vector<SMART_PTR_NS::shared_ptr<MeetingInfo> >();
				pMeetingVec->push_back(sharedObj);

				SMART_PTR_NS::shared_ptr<std::vector<SMART_PTR_NS::shared_ptr<MeetingInfo> > > meetingList;
				meetingList.reset(pMeetingVec);

				SMART_PTR_NS::shared_ptr<MeetingTime> meetingDay;

				m_meetingCallback->onMeetingListed(meetingDay, meetingList, MeetingErrorCodeEnum::MEETING_SUCCESS);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnListOneDayEvents, this, strEvents),
					"MeetingMgrControllerImpl::OnListOneDayEvents");
			}
		}
	};

	void MeetingMgrControllerImpl::OnInitializeOIFailed()
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnInitializeOIFailed:");

		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onInitializeOIFailed();
			}
			else
			{
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::OnInitializeOIFailed, this), "MeetingMgrControllerImpl::OnInitializeOIFailed");
			}
		}
	}

	void MeetingMgrControllerImpl::OnInitializeResult(LONG const &lResult)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnInitializeResult:");

		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onInitializeResult(lResult);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::OnInitializeResult, this, lResult), "MeetingMgrControllerImpl::OnInitializeResult");
			}
		}
	}

	void MeetingMgrControllerImpl::OnMeetingStart(std::wstring const& strEventID, std::wstring const& strMeetingKey)
	{
		string strEventID_ = JCFCoreUtils::toString(strEventID);
		string strMeetingKey_ = JCFCoreUtils::toString(strMeetingKey);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingStart:eventId: " << strEventID_.c_str()
			<< " meetingkey: " << strMeetingKey_.c_str());

		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingStart(strEventID_, strMeetingKey_);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::OnMeetingStart, this, strEventID, strMeetingKey), "MeetingMgrControllerImpl::OnMeetingStart");
			}
		}

	};

	void MeetingMgrControllerImpl::OnMeetingEnd(std::wstring const& strEventID, std::wstring const& strMeetingKey)
	{
		std::string strEventID_ = JCFCoreUtils::toString(strEventID);
		std::string strMeetingKey_ = JCFCoreUtils::toString(strMeetingKey);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingEnd:eventId: " << strEventID_.c_str()
			<< " meetingkey: " << strMeetingKey_.c_str());

		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingEnd(strEventID_, strMeetingKey_);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::OnMeetingEnd, this, strEventID, strMeetingKey), "MeetingMgrControllerImpl::OnMeetingEnd");
			}
		}

	};

	void MeetingMgrControllerImpl::OnMeetingReminderHappened(JabberMeeting::MeetingReminderItem const& objMeetingReminderItem)
	{
		MeetingReminderInfoBaseImpl * pObj = new MeetingReminderInfoBaseImpl(unifiedFactory);
		ConvertMeetingReminderInfo2Object(objMeetingReminderItem, *pObj);

		SMART_PTR_NS::shared_ptr<MeetingReminderInfoBaseImpl> sharedObj;
		sharedObj.reset(pObj);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingReminderHappened");

		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingReminderHappened(sharedObj);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingReminderHappened, this, objMeetingReminderItem),
					"MeetingMgrControllerImpl::OnMeetingReminderHappened");
			}
		}
	}

	void MeetingMgrControllerImpl::OnMeetingReminderEnded(std::wstring const & strEventID)
	{
		std::string strEventID_ = JCFCoreUtils::toString(strEventID);
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingReminderEnded:eventId: " << strEventID_.c_str());

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingReminderEnded(strEventID_);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingReminderEnded, this, strEventID),
					"MeetingMgrControllerImpl::OnMeetingReminderEnded");
			}
		}
	}

	void MeetingMgrControllerImpl::OnMeetingIMAlertHappened(JabberMeeting::MeetingIMAlertItem const &obj)
	{
		MeetingIMAlertItemObjectBaseImpl * pObj = new MeetingIMAlertItemObjectBaseImpl(unifiedFactory);
		ConvertMeetingIMAlertItem2Object(obj, *pObj);

		SMART_PTR_NS::shared_ptr<MeetingIMAlertItemObjectBaseImpl> sharedObj;
		sharedObj.reset(pObj);
		
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingIMAlertHappened");

		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingIMAlertHappened(sharedObj);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingIMAlertHappened, this, obj),
					"MeetingMgrControllerImpl::OnMeetingIMAlertHappened");
			}
		}
	}

	void MeetingMgrControllerImpl::OnOvernight(JabberMeeting::MeetingMgrErrorCode eRet)
	{
		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnOvernight: " << eRet);

		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				int errCode = eRet;
				m_meetingCallback->onMeetingOvernight();
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnOvernight, this, eRet),
					"MeetingMgrControllerImpl::OnOvernight");
			}
		}
	}

	void MeetingMgrControllerImpl::OnGetWbxMeetingDetail(std::wstring const& strEventXML, std::wstring const& strMeetingDetail)
	{
		std::string strContent = JCFCoreUtils::toString(strEventXML);
		std::string strDetail = JCFCoreUtils::toString(strMeetingDetail);
		
		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				// Mingfeng 2015.8.28: strEvents is XML content for all meetings to list. To unify meeting 
				// service interface and be compatible with UI layer code, temporarily set the content to newly
				// added MeetingEventXML in the first node of meetingList vector.
				MeetingInfoImpl * pObj = new MeetingInfoImpl(unifiedFactory);
				pObj->setMeetingEventXML(strContent);
				pObj->setMeetingDetailXML(strDetail);
				SMART_PTR_NS::shared_ptr<MeetingInfoImpl> sharedObj;
				sharedObj.reset(pObj);

				m_meetingCallback->onGetMeetingDetail(sharedObj);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnGetWbxMeetingDetail, this, strEventXML, strMeetingDetail),
					"MeetingMgrControllerImpl::OnGetWbxMeetingDetail");
			}
		}
	}

	void MeetingMgrControllerImpl::OnGetCMRInfo(const std::wstring& cmrInfo)
	{
		std::string strCMRInfo = JCFCoreUtils::toString(cmrInfo);
		
		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onGetCMRInfo(strCMRInfo);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnGetCMRInfo, this, cmrInfo),
					"MeetingMgrControllerImpl::OnGetCMRInfo");
			}
		}
	}

	void MeetingMgrControllerImpl::ConvertSystemTimeToMeetingTime(const SYSTEMTIME& sysTime, MeetingTimeBaseImpl* pMeetingTime)
	{
		pMeetingTime->setYear(sysTime.wYear);
		pMeetingTime->setMonth(sysTime.wMonth);
		pMeetingTime->setDay(sysTime.wDay);
		pMeetingTime->setHour(sysTime.wHour);
		pMeetingTime->setMinute(sysTime.wMinute);
	}

	void MeetingMgrControllerImpl::ConvertMeetingReminderInfo2Object(const JabberMeeting::MeetingReminderItem &objMeetingReminderItem, MeetingReminderInfoBaseImpl &obj)
	{
		std::string strMeetingEventID = JCFCoreUtils::toString(objMeetingReminderItem.strMeetingEventID);
		std::string strMeetingTitle = JCFCoreUtils::toString(objMeetingReminderItem.strMeetingTitle);
		std::string strLocation = JCFCoreUtils::toString(objMeetingReminderItem.strLocation);
		std::string strHostEmail = JCFCoreUtils::toString(objMeetingReminderItem.strHostEmail);
		std::string strHostDisplayName = JCFCoreUtils::toString(objMeetingReminderItem.strHostDisplayName);

		MeetingTimeBaseImpl * pStartMeetingTime = new MeetingTimeBaseImpl(m_unifiedFactory);
		ConvertSystemTimeToMeetingTime(objMeetingReminderItem.stStartTime, pStartMeetingTime);
		SMART_PTR_NS::shared_ptr<MeetingTimeBaseImpl> sharedStartTime;
		sharedStartTime.reset(pStartMeetingTime);

		MeetingTimeBaseImpl * pEndMeetingTime = new MeetingTimeBaseImpl(m_unifiedFactory);
		ConvertSystemTimeToMeetingTime(objMeetingReminderItem.stEndTime, pEndMeetingTime);
		SMART_PTR_NS::shared_ptr<MeetingTimeBaseImpl> sharedEndTime;
		sharedEndTime.reset(pEndMeetingTime);

		obj.setMeetingID(strMeetingEventID);
		obj.setMeetingTopic(strMeetingTitle);
		obj.setLocation(strLocation);
		obj.setHostEmail(strHostEmail);
		obj.setHostDisplayName(strHostDisplayName);
		obj.setIsHost(objMeetingReminderItem.bIsHost);

		obj.setStartTime(sharedStartTime);
		obj.setEndTime(sharedEndTime);

		obj.setDuration(objMeetingReminderItem.lDuration);
	}

	void MeetingMgrControllerImpl::ConvertMeetingIMAlertItem2Object(const JabberMeeting::MeetingIMAlertItem &objItem, MeetingIMAlertItemObjectBaseImpl & obj)
	{
		std::string strMeetingKey = JCFCoreUtils::toString(objItem.strMeetingKey);
		std::string strMeetingTitle = JCFCoreUtils::toString(objItem.strMeetingTitle);
		std::string strDetailMeetingInfo = JCFCoreUtils::toString(objItem.strDetailMeetingInfo);
		std::string strHostEmail = JCFCoreUtils::toString(objItem.strHostEmail);
		std::string strHostDisplayName = JCFCoreUtils::toString(objItem.strHostDisplayName);
		std::string strAttendeeInfo = JCFCoreUtils::toString(objItem.strAttendeeInfo);
		std::string strJoinURL = JCFCoreUtils::toString(objItem.strJoinURL);
		std::string strSiteURL = JCFCoreUtils::toString(objItem.strSiteURL);

		obj.setstrMeetingKey(strMeetingKey);
		obj.setstrMeetingTitle(strMeetingTitle);
		obj.setstrAttendeeInfo(strAttendeeInfo);
		obj.setstrHostEmail(strHostEmail);
		obj.setstrHostDisplayName(strHostDisplayName);
		obj.setstrJoinURL(strJoinURL);
		obj.setstrDetailMeetingInfo(strDetailMeetingInfo);
		obj.setstrSiteURL(strSiteURL);
	}

#pragma region IInstantMeetingEvent
	void MeetingMgrControllerImpl::OnMeetingStart(std::wstring const& strMeetingKey, std::wstring const& strSessionName, BOOL bHost)
	{
		csf::ScopedReadRWLock lock(mLock);

		std::string strMeetingKey_ = JCFCoreUtils::toString(strMeetingKey);
		std::string strSessionName_ = JCFCoreUtils::toString(strSessionName);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingStart");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingStarted(strSessionName_, strMeetingKey_, bHost == TRUE ? true : false);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingStart, this, strMeetingKey, strSessionName, bHost),
					"MeetingMgrControllerImpl::OnMeetingStart");
			}
		}
	}

	void MeetingMgrControllerImpl::OnMeetingError(std::wstring const& strMeetingKey, std::wstring const& strSessionName, BOOL bHost)
	{
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingError");

		if (NULL == m_pIMMgr)
		{
			return;
		}

		std::wstring strTemp;
		int nErrCode = m_pIMMgr->GetLastError(strTemp);

		std::string strMeetingKey_ = JCFCoreUtils::toString(strMeetingKey);
		std::string strSessionName_ = JCFCoreUtils::toString(strSessionName);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingError(strSessionName_, nErrCode);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingError, this, strMeetingKey, strSessionName, bHost),
					"MeetingMgrControllerImpl::OnMeetingError");
			}
		}
	}

	void MeetingMgrControllerImpl::OnMeetingEnd(std::wstring const& strMeetingKey, std::wstring const& strSessionName, BOOL bHost)
	{
		csf::ScopedReadRWLock lock(mLock);

		std::string strMeetingKey_ = JCFCoreUtils::toString(strMeetingKey);
		std::string strSessionName_ = JCFCoreUtils::toString(strSessionName);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingEnd");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingEnded(strSessionName_, strMeetingKey_, bHost == TRUE ? true : false);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingEnd, this, strMeetingKey, strSessionName, bHost),
					"MeetingMgrControllerImpl::OnMeetingEnd");
			}
		}
	}

	void MeetingMgrControllerImpl::OnMeetingInvitation(std::wstring const& strSessionName)
	{
		csf::ScopedReadRWLock lock(mLock);
		std::string strSessionName_ = JCFCoreUtils::toString(strSessionName);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingInvitation");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingInvitationReceived(strSessionName_, "");
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingInvitation, this, strSessionName),
					"MeetingMgrControllerImpl::OnMeetingInvitation");
			}
		}
	}

	void MeetingMgrControllerImpl::OnMeetingCancelInvitation(std::wstring const& strSessionName)
	{
		csf::ScopedReadRWLock lock(mLock);
		std::string strSessionName_ = JCFCoreUtils::toString(strSessionName);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingCancelInvitation");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingInvitationCanceled(strSessionName_);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingCancelInvitation, this, strSessionName),
					"MeetingMgrControllerImpl::OnMeetingCancelInvitation");
			}
		}
	}

	void MeetingMgrControllerImpl::OnMeetingDecline(std::wstring const& strSessionName, std::wstring const& strBuddyName)
	{
		csf::ScopedReadRWLock lock(mLock);
		std::string strSessionName_ = JCFCoreUtils::toString(strSessionName);
		std::string strBuddyName_ = JCFCoreUtils::toString(strBuddyName);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnMeetingDecline");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onMeetingDeclined(strSessionName_, strBuddyName_);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnMeetingDecline, this, strSessionName, strBuddyName),
					"MeetingMgrControllerImpl::OnMeetingDecline");
			}
		}
	}

	void MeetingMgrControllerImpl::OnPopupWaitDialog()
	{
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnPopupWaitDialog");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onShowMeetingWaitDialog();
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnPopupWaitDialog, this),
					"MeetingMgrControllerImpl::OnPopupWaitDialog");
			}
		}
	}

	void MeetingMgrControllerImpl::OnCloseWaitDialog()
	{
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnCloseWaitDialog");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onHideMeetingWaitDialog();
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnCloseWaitDialog, this),
					"MeetingMgrControllerImpl::OnCloseWaitDialog");
			}
		}
	}

	void MeetingMgrControllerImpl::OnShowErrorDialog(int nErrorCode)
	{
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnShowErrorDialog");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				m_meetingCallback->onShowMeetingErrorDialog(nErrorCode);
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnShowErrorDialog, this, nErrorCode),
					"MeetingMgrControllerImpl::OnShowErrorDialog");
			}
		}
	}

	CMRMeetingData MeetingMgrControllerImpl::createCMRMeetingData(const CMR_MeetingData& meetingData)
	{
		CMRMeetingData ret;
		ret.meetingKey = JCFCoreUtils::toString(meetingData.strMeetingKey);
		ret.mtgtk = JCFCoreUtils::toString(meetingData.strMTGTK);
		ret.sipUrl = JCFCoreUtils::toString(meetingData.strCMRSipURL);
		ret.hostKey = JCFCoreUtils::toString(meetingData.strHostKey);
		ret.joinUrl = JCFCoreUtils::toString(meetingData.strCMRJoinURL);

		return ret;
	}

	void MeetingMgrControllerImpl::OnCMRMeetingStarted(const CMR_MeetingData & info)
	{
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnCMRMeetingStarted");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				CSFUnified::CMRMeetingData meetingData = createCMRMeetingData(info);
				CMRControllerMgr * pMgr = MeetingServiceObjectManager::getInstance().getCMRMgr();
				if (NULL != pMgr)
				{
					pMgr->onCmrMeetingStarted(meetingData);
				}
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnCMRMeetingStarted, this, info),
					"MeetingMgrControllerImpl::OnCMRMeetingStarted");
			}
		}
	}

	void MeetingMgrControllerImpl::OnCMRMeetingJoined(const CMR_MeetingData & info)
	{
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnCMRMeetingJoined");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				CMRControllerMgr * pMgr = MeetingServiceObjectManager::getInstance().getCMRMgr();
				if (NULL != pMgr)
				{
					pMgr->onCmrMeetingJoined();
				}
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnCMRMeetingJoined, this, info),
					"MeetingMgrControllerImpl::OnCMRMeetingJoined");
			}
		}
	}

	void MeetingMgrControllerImpl::OnCMRMeetingFailed(int nErrorCode)
	{
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnCMRMeetingFailed");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				CMRControllerMgr * pMgr = MeetingServiceObjectManager::getInstance().getCMRMgr();
				if (NULL != pMgr)
				{
					pMgr->onCmrMeetingError();
				}
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnCMRMeetingFailed, this, nErrorCode),
					"MeetingMgrControllerImpl::OnCMRMeetingFailed");
			}
		}
	}

	void MeetingMgrControllerImpl::OnCMRMeetingEnded()
	{
		csf::ScopedReadRWLock lock(mLock);

		CSFLogDebugS(MeetingMgrControllerImplLogger, "MeetingMgrControllerImpl::OnCMRMeetingEnded");

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
		if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
		{
			bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
			if (bOnDispatcher)
			{
				CMRControllerMgr * pMgr = MeetingServiceObjectManager::getInstance().getCMRMgr();
				if (NULL != pMgr)
				{
					pMgr->onCmrMeetingEnded();
				}
			}
			else
			{
				pServiceDispatcher->enqueueBlock(
					boost::bind(&MeetingMgrControllerImpl::OnCMRMeetingEnded, this),
					"MeetingMgrControllerImpl::OnCMRMeetingEnded");
			}
		}
	}
#pragma endregion
};


#endif